// pages/api/auth/register.ts
import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import * as bcrypt from "bcryptjs";
import { z } from "zod";

// Simple validator
const Input = z.object({
  email: z.string().email(),
  password: z.string().min(6).max(128),
  name: z.string().min(1).max(100).optional(),
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  if (req.method !== "POST")
    return res.status(405).json({ ok: false, error: "Method not allowed" });

  const parsed = Input.safeParse(req.body);
  if (!parsed.success) {
    return res.status(400).json({
      ok: false,
      error: "Invalid input",
      details: parsed.error.flatten(),
    });
  }

  const { email, password, name } = parsed.data;

  try {
    // Prevent duplicate emails
    const exists = await prisma.user.findUnique({ where: { email } });
    if (exists)
      return res.status(409).json({ ok: false, error: "Email already in use" });

    // Hash password
    const passwordHash = await bcrypt.hash(password, 10);

    // Create user (role = "USER" by default)
    const user = await prisma.user.create({
      data: { email, name: name ?? null, passwordHash, role: "USER" },
      select: {
        id: true,
        email: true,
        name: true,
        role: true,
        createdAt: true,
      },
    });

    return res.status(201).json({ ok: true, user });
  } catch (err: any) {
    console.error(err);
    return res.status(500).json({ ok: false, error: "Registration failed" });
  }
}
